<?php
/**
 * FastUnlock Top Banner (Dynamic Placement Edition)
 * @package   fastunlocktopbanner
 * @author    Fast Unlocking
 * @version   2.4.0
 * @license   AFL-3.0
 */
if (!defined('_PS_VERSION_')) { exit; }

class Fastunlocktopbanner extends Module
{
    const CFG_ENABLED    = 'FUTB_ENABLED';
    const CFG_LINES      = 'FUTB_LINES';
    const CFG_BG_COLOR   = 'FUTB_BG_COLOR';
    const CFG_TEXT_COLOR = 'FUTB_TEXT_COLOR';
    const CFG_HEIGHT     = 'FUTB_HEIGHT';
    const CFG_HOLD       = 'FUTB_HOLD_SECONDS';
    const CFG_LOOP       = 'FUTB_LOOP';
    const CFG_FONT       = 'FUTB_FONT_SIZE';
    const CFG_POSITION   = 'FUTB_POSITION';

    // ---------------- Licensing (FastUnlocking License Server) ----------------
    const CONF_SERVER_URL = 'FUTB_LICENSE_SERVER_URL';
    const CONF_LICENSE_KEY = 'FUTB_LICENSE_KEY';
    const CONF_LICENSE_OK = 'FUTB_LICENSE_OK';
    const CONF_LICENSE_LASTCHECK = 'FUTB_LICENSE_LASTCHECK';
    const CONF_AUTO_DISABLED = 'FUTB_AUTO_DISABLED';
    const MODULE_CODE = 'fastunlocktopbanner';
    const DEFAULT_SERVER_URL = 'https://fastunlocking.com/server';
    const LICENSE_CACHE_TTL = 300; // 5 minutes

    public function __construct()
    {
        $this->name = 'fastunlocktopbanner';
        $this->tab = 'front_office_features';
        $this->version = '2.5.0';
        $this->author = 'Fast Unlocking';
        $this->bootstrap = true;
        $this->ps_versions_compliancy = array('min' => '1.7.0.0', 'max' => _PS_VERSION_);
        parent::__construct();
        $this->displayName = $this->l('FastUnlock Top Banner (Dynamic Placement)');
        $this->description = $this->l('Top banner with fade rotation. Choose position (Top / Above login / Below nav / Above footer), adjust height, font, colors, hold.');
    }

    public function install()
{
    $default_lines = "🔥 Fast Unlocking: Adjustable banner height!\n📢 Second message line goes here";

    $ok = parent::install()
        && $this->registerHook('displayTop')
        && $this->registerHook('displayBanner')
        && $this->registerHook('displayNav1')
        && $this->registerHook('displayFooterBefore')
        && $this->registerHook('header')
        && Configuration::updateValue(self::CFG_ENABLED, 1)
        && Configuration::updateValue(self::CFG_LINES, $default_lines, true)
        && Configuration::updateValue(self::CFG_BG_COLOR, '#cc0000')
        && Configuration::updateValue(self::CFG_TEXT_COLOR, '#ffffff')
        && Configuration::updateValue(self::CFG_HEIGHT, 56)
        && Configuration::updateValue(self::CFG_HOLD, 5)
        && Configuration::updateValue(self::CFG_LOOP, 1)
        && Configuration::updateValue(self::CFG_FONT, 16)
        && Configuration::updateValue(self::CFG_POSITION, 'displayBanner')
        && Configuration::updateValue(self::CONF_SERVER_URL, self::DEFAULT_SERVER_URL)
        && Configuration::updateValue(self::CONF_LICENSE_KEY, '')
        && Configuration::updateValue(self::CONF_LICENSE_OK, 0)
        && Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, '')
        && Configuration::updateValue(self::CONF_AUTO_DISABLED, 0);

    // Enforce initial state immediately (no key => module auto-disables)
    if ($ok) {
        $this->ensureLicenseFresh();
    }

    return $ok;
}

    public function uninstall()
    {
        return Configuration::deleteByName(self::CFG_ENABLED)
            && Configuration::deleteByName(self::CFG_LINES)
            && Configuration::deleteByName(self::CFG_BG_COLOR)
            && Configuration::deleteByName(self::CFG_TEXT_COLOR)
            && Configuration::deleteByName(self::CFG_HEIGHT)
            && Configuration::deleteByName(self::CFG_HOLD)
            && Configuration::deleteByName(self::CFG_LOOP)
            && Configuration::deleteByName(self::CFG_FONT)
            && Configuration::deleteByName(self::CFG_POSITION)

            // Licensing
            && Configuration::deleteByName(self::CONF_SERVER_URL)
            && Configuration::deleteByName(self::CONF_LICENSE_KEY)
            && Configuration::deleteByName(self::CONF_LICENSE_OK)
            && Configuration::deleteByName(self::CONF_LICENSE_LASTCHECK)
            && parent::uninstall();
    }

    // ---------------- Licensing helpers ----------------
    protected function shopDomain(): string
    {
        $d = (string)Tools::getShopDomain(false);
        if ($d === '') {
            $d = (string)Tools::getShopDomainSsl(false);
        }
        return $this->normalizeDomain($d);
    }

    protected function normalizeDomain(string $d): string
    {
        $d = trim(strtolower($d));
        $d = preg_replace('#^https?://#', '', $d);
        $d = preg_replace('#^www\.#', '', $d);
        $d = preg_replace('#/.*$#', '', $d);
        $d = preg_replace('#:\d+$#', '', $d);
        return (string)$d;
    }

    /**
     * Ensure license status is refreshed at most every 12 hours.
     * This prevents slow BO loads or repeated remote calls when license is invalid.
     */
    /**
 * Ensure license status is refreshed at most every LICENSE_CACHE_TTL seconds.
 * Enforces: no key / invalid / revoked => module auto-disables.
 * If it becomes valid again, module is auto-enabled only if it was auto-disabled by licensing.
 */
protected function ensureLicenseFresh(): void
{
    $key = trim((string) Configuration::get(self::CONF_LICENSE_KEY));
    $last = (string) Configuration::get(self::CONF_LICENSE_LASTCHECK);

    // TTL guard
    if ($last !== '') {
        $ts = strtotime($last);
        if ($ts && (time() - $ts) < (int) self::LICENSE_CACHE_TTL) {
            return;
        }
    }

    // No key => immediately consider unlicensed + auto-disable
    if ($key === '') {
        Configuration::updateValue(self::CONF_LICENSE_OK, 0);
        Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, date('Y-m-d H:i:s'));

        if ($this->active) {
            Configuration::updateValue(self::CONF_AUTO_DISABLED, 1);
            $this->disable();
        }
        return;
    }

    try {
        $res = $this->validateLicenseRemote($key);
        $ok = !empty($res['ok']) ? 1 : 0;

        Configuration::updateValue(self::CONF_LICENSE_OK, $ok);
        Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, date('Y-m-d H:i:s'));

        if ($ok === 0) {
            // Invalid / revoked => disable now
            if ($this->active) {
                Configuration::updateValue(self::CONF_AUTO_DISABLED, 1);
                $this->disable();
            }
        } else {
            // Valid => only re-enable if we auto-disabled it before
            $auto = (int) Configuration::get(self::CONF_AUTO_DISABLED);
            if ($auto === 1 && !$this->active) {
                $this->enable();
                Configuration::updateValue(self::CONF_AUTO_DISABLED, 0);
            }
        }
    } catch (\Throwable $e) {
        // Never break BO due to license server issues.
        Configuration::updateValue(self::CONF_LICENSE_OK, 0);
        Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, date('Y-m-d H:i:s'));

        if ($this->active) {
            Configuration::updateValue(self::CONF_AUTO_DISABLED, 1);
            $this->disable();
        }
    }
}


    protected function isLicensed(): bool
    {
        $key = trim((string)Configuration::get(self::CONF_LICENSE_KEY));
        if ($key === '') {
            return false;
        }

        // Cache license checks for 12 hours to reduce server load.
        $ok = (int)Configuration::get(self::CONF_LICENSE_OK);
        $last = (string)Configuration::get(self::CONF_LICENSE_LASTCHECK);
        if ($ok === 1 && $last !== '') {
            $ts = strtotime($last);
            if ($ts && (time() - $ts) < 5 * 60) {
                return true;
            }
        }

        $res = $this->validateLicenseRemote($key);
        Configuration::updateValue(self::CONF_LICENSE_OK, $res['ok'] ? 1 : 0);
        Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, date('Y-m-d H:i:s'));
        return (bool)$res['ok'];
    }

    protected function validateLicenseRemote(string $key): array
    {
        $server = trim((string)Configuration::get(self::CONF_SERVER_URL));
        if ($server === '') {
            $server = self::DEFAULT_SERVER_URL;
            Configuration::updateValue(self::CONF_SERVER_URL, $server);
        }
        $server = rtrim($server, '/');
        $domain = $this->shopDomain();

        $url = $server . '/api/validate.php?license_key=' . rawurlencode($key)
            . '&module_code=' . rawurlencode($this->name)
            . '&domain=' . rawurlencode($domain);

        try {
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 8);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
            $body = curl_exec($ch);
            $errno = curl_errno($ch);
            $err = curl_error($ch);
            $http = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($errno) {
                return ['ok' => 0, 'message' => 'cURL: ' . $err, 'http' => $http];
            }
            if ($http < 200 || $http >= 300) {
                return ['ok' => 0, 'message' => 'HTTP ' . $http, 'http' => $http, 'raw' => (string)$body];
            }

            $json = json_decode((string)$body, true);
            if (!is_array($json)) {
                return ['ok' => 0, 'message' => 'Bad JSON from license server', 'http' => $http, 'raw' => (string)$body];
            }
            $ok = !empty($json['ok']);
            $msg = (string)($json['message'] ?? ($ok ? 'OK' : 'License invalid'));
            return ['ok' => $ok ? 1 : 0, 'message' => $msg, 'data' => $json];
        } catch (Throwable $e) {
            return ['ok' => 0, 'message' => 'SERVER_ERROR'];
        }
    }

    public function hookHeader()
    {
        $this->context->controller->registerStylesheet(
            'module-fastunlocktopbanner-front',
            'modules/'.$this->name.'/views/css/front.css',
            array('media' => 'all', 'priority' => 50)
        );
        $this->context->controller->registerJavascript(
            'module-fastunlocktopbanner-front',
            'modules/'.$this->name.'/views/js/front.js',
            array('position' => 'bottom', 'priority' => 50)
        );
        $height = (int)Configuration::get(self::CFG_HEIGHT);
        $font   = (int)Configuration::get(self::CFG_FONT);
        $css = ':root{--futb-height:' . $height . 'px;--futb-font:' . $font . 'px;}';
        $this->context->controller->addCSS('data:text/css;base64,'.base64_encode($css), 'all', null, true);
        return '';
    }

    public function hookDisplayTop($params){ return $this->renderAt('displayTop'); }
    public function hookDisplayBanner($params){ return $this->renderAt('displayBanner'); }
    public function hookDisplayNav1($params){ return $this->renderAt('displayNav1'); }
    public function hookDisplayFooterBefore($params){ return $this->renderAt('displayFooterBefore'); }

    private function renderAt($hook)
    {
        // Keep front-office output strictly gated by licensing.
        // This also enforces license revocation without needing the BO config page.
        $this->ensureLicenseFresh();

        if (!(int)Configuration::get(self::CFG_ENABLED)) return '';
        $licenseKey = trim((string)Configuration::get(self::CONF_LICENSE_KEY));
        if ($licenseKey === '') return '';
        if (!$this->isLicensed()) return '';
        $selected = Configuration::get(self::CFG_POSITION);
        if ($selected !== $hook) return '';

        $lines = preg_split("/\r\n|\n|\r/", (string)Configuration::get(self::CFG_LINES));
        $clean = array();
        foreach ($lines as $ln) { if (trim($ln) !== '') { $clean[] = $ln; } }
        if (empty($clean)) return '';

        $this->context->smarty->assign(array(
            'futb_bg_color'   => Configuration::get(self::CFG_BG_COLOR),
            'futb_text_color' => Configuration::get(self::CFG_TEXT_COLOR),
            'futb_lines_json' => json_encode($clean),
            'futb_hold'       => (int)Configuration::get(self::CFG_HOLD),
            'futb_loop'       => (int)Configuration::get(self::CFG_LOOP),
            'futb_position'   => $selected,
        ));
        return $this->display(__FILE__, 'views/templates/hook/topbanner.tpl');
    }

    public function getContent()
    {
        $output = '';

        // --- License save/validate ---
        if (Tools::isSubmit('submitFutbLicense')) {
            $key = trim((string)Tools::getValue('futb_license_key'));
            Configuration::updateValue(self::CONF_SERVER_URL, self::DEFAULT_SERVER_URL);
            Configuration::updateValue(self::CONF_LICENSE_KEY, $key);

            if ($key === '') {
                Configuration::updateValue(self::CONF_LICENSE_OK, 0);
                $output .= $this->displayError($this->l('Please enter a license key.'));
            } else {
                $res = $this->validateLicenseRemote($key);
                if (!empty($res['ok'])) {
                    Configuration::updateValue(self::CONF_LICENSE_OK, 1);
                    Configuration::updateValue(self::CONF_LICENSE_LASTCHECK, time());
                    $output .= $this->displayConfirmation($this->l('Licensed')); // keep short
                } else {
                    Configuration::updateValue(self::CONF_LICENSE_OK, 0);
                    $msg = isset($res['message']) ? (string)$res['message'] : 'License check failed';
                    $output .= $this->displayError($this->l('License invalid: ') . $msg);
                }
            }
        }

        if (Tools::isSubmit('submitFastUnlockTopBanner')) {
            Configuration::updateValue(self::CFG_ENABLED, (int)Tools::getValue(self::CFG_ENABLED, 0));
            Configuration::updateValue(self::CFG_LINES, Tools::getValue(self::CFG_LINES, ''), true);
            Configuration::updateValue(self::CFG_BG_COLOR, Tools::getValue(self::CFG_BG_COLOR, '#cc0000'));
            Configuration::updateValue(self::CFG_TEXT_COLOR, Tools::getValue(self::CFG_TEXT_COLOR, '#ffffff'));
            Configuration::updateValue(self::CFG_HEIGHT, (int)Tools::getValue(self::CFG_HEIGHT, 56));
            Configuration::updateValue(self::CFG_HOLD, (int)Tools::getValue(self::CFG_HOLD, 5));
            Configuration::updateValue(self::CFG_LOOP, (int)Tools::getValue(self::CFG_LOOP, 1));
            Configuration::updateValue(self::CFG_FONT, (int)Tools::getValue(self::CFG_FONT, 16));
            $pos = Tools::getValue(self::CFG_POSITION, 'displayBanner');
            if (!in_array($pos, array('displayTop','displayBanner','displayNav1','displayFooterBefore'))) {
                $pos = 'displayBanner';
            }
            Configuration::updateValue(self::CFG_POSITION, $pos);

            if (class_exists('Media')) Media::clearCache();
            if (class_exists('Tools')) Tools::clearCache();

            $output .= $this->displayConfirmation($this->l('Settings saved.'));
        }
        // License status banner (auto-refreshes every 12h)
        $this->ensureLicenseFresh();
        $licenseKey = trim((string)Configuration::get(self::CONF_LICENSE_KEY));
        if ($licenseKey === '') {
            $output .= $this->displayWarning($this->l('FastUnlocking License: not activated (enter your serial and click Activate).'));
        } elseif (!$this->isLicensed()) {
            $output .= $this->displayWarning($this->l('FastUnlocking License: License check failed'));
        }

        return $output . $this->renderLicenseForm() . $this->renderForm();
    }

    protected function renderLicenseForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->default_form_language = (int)Configuration::get('PS_LANG_DEFAULT');
        $helper->allow_employee_form_lang = (int)Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG');
        $helper->identifier = 'id_module';
        $helper->submit_action = 'submitFutbLicense';
        $helper->currentIndex = AdminController::$currentIndex . '&configure=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $key = (string)Configuration::get(self::CONF_LICENSE_KEY);
        $domain = $this->shopDomain();

        $helper->fields_value = array(
            'futb_license_key' => $key,
        );

        $fields_form = array(
            'form' => array(
                'legend' => array('title' => $this->l('License'), 'icon' => 'icon-key'),
                'description' => $this->l('This license will be locked to this domain: ') . $domain,
                'input' => array(
                    array(
                        'type' => 'text',
                        'label' => $this->l('License Key'),
                        'name' => 'futb_license_key',
                        'required' => true,
                        'desc' => $this->l('Enter the key you received after purchase.'),
                    ),
                ),
                'submit' => array('title' => $this->l('Validate License')),
            )
        );

        return $helper->generateForm(array($fields_form));
    }

    protected function renderForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->default_form_language = (int)Configuration::get('PS_LANG_DEFAULT');
        $helper->allow_employee_form_lang = (int)Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG');
        $helper->identifier = 'id_module';
        $helper->submit_action = 'submitFastUnlockTopBanner';
        $helper->currentIndex = AdminController::$currentIndex . '&configure=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $positions = array(
            array('id'=>'displayTop','name'=>$this->l('Top of page')),
            array('id'=>'displayBanner','name'=>$this->l('Above login bar')),
            array('id'=>'displayNav1','name'=>$this->l('Below navigation')),
            array('id'=>'displayFooterBefore','name'=>$this->l('Above footer')),
        );

        $helper->fields_value = array(
            self::CFG_ENABLED    => (int)Configuration::get(self::CFG_ENABLED),
            self::CFG_LINES      => Configuration::get(self::CFG_LINES),
            self::CFG_BG_COLOR   => Configuration::get(self::CFG_BG_COLOR),
            self::CFG_TEXT_COLOR => Configuration::get(self::CFG_TEXT_COLOR),
            self::CFG_HEIGHT     => (int)Configuration::get(self::CFG_HEIGHT),
            self::CFG_HOLD       => (int)Configuration::get(self::CFG_HOLD),
            self::CFG_LOOP       => (int)Configuration::get(self::CFG_LOOP),
            self::CFG_FONT       => (int)Configuration::get(self::CFG_FONT),
            self::CFG_POSITION   => Configuration::get(self::CFG_POSITION),
        );

        $fields_form = array(
            'form' => array(
                'legend' => array('title' => $this->displayName . ' v' . $this->version, 'icon' => 'icon-cogs'),
                'input'  => array(
                    array('type'=>'switch','label'=>$this->l('Enable banner'),'name'=>self::CFG_ENABLED,'is_bool'=>true,'values'=>array(
                        array('id'=>'on','value'=>1,'label'=>$this->l('Enabled')), array('id'=>'off','value'=>0,'label'=>$this->l('Disabled'))
                    )),
                    array('type'=>'textarea','label'=>$this->l('Banner lines (one per line, HTML allowed)'),'name'=>self::CFG_LINES,'autoload_rte'=>true,'cols'=>60,'rows'=>5),
                    array('type'=>'color','label'=>$this->l('Background color'),'name'=>self::CFG_BG_COLOR),
                    array('type'=>'color','label'=>$this->l('Text color'),'name'=>self::CFG_TEXT_COLOR),
                    array('type'=>'text','label'=>$this->l('Banner height (px)'),'name'=>self::CFG_HEIGHT,'class'=>'fixed-width-xs','suffix'=>'px'),
                    array('type'=>'text','label'=>$this->l('Font size (px)'),'name'=>self::CFG_FONT,'class'=>'fixed-width-xs','suffix'=>'px'),
                    array('type'=>'text','label'=>$this->l('Hold (seconds)'),'name'=>self::CFG_HOLD,'class'=>'fixed-width-xs','suffix'=>'s'),
                    array('type'=>'switch','label'=>$this->l('Loop messages'),'name'=>self::CFG_LOOP,'is_bool'=>true,'values'=>array(
                        array('id'=>'yes','value'=>1,'label'=>$this->l('Yes')), array('id'=>'no','value'=>0,'label'=>$this->l('No'))
                    )),
                    array('type'=>'select','label'=>$this->l('Banner position (hook)'),'name'=>self::CFG_POSITION,
                          'options'=>array('query'=>$positions,'id'=>'id','name'=>'name')),
                ),
                'submit' => array('title' => $this->l('Save')),
            )
        );
        return $helper->generateForm(array($fields_form));
    }
}
